package jcircus.gui;

import java.io.FileWriter;
import java.io.IOException;
import java.io.PrintStream;
import java.util.Map;
import java.util.List;
import java.awt.Container;
import java.awt.Dimension;
import java.awt.GridBagConstraints;
import java.awt.GridBagLayout;
import java.awt.Insets;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.io.File;
import javax.swing.BorderFactory;
import javax.swing.JButton;
import javax.swing.JComponent;
import javax.swing.JFileChooser;
import javax.swing.JFrame;
import javax.swing.JLabel;
import javax.swing.JOptionPane;
import javax.swing.JPanel;
import javax.swing.JScrollPane;
import javax.swing.JTextArea;
import javax.swing.JTextField;
import javax.swing.event.DocumentEvent;
import javax.swing.event.DocumentListener;
import javax.swing.text.PlainDocument;
import jcircus.JCircus;
import jcircus.exceptions.FailParsingException;
import jcircus.exceptions.FailTranslationException;
import jcircus.exceptions.FailTypeCheckingException;
import jcircus.exceptions.JCircusException;
import jcircus.util.Constants;
import jcircus.util.ProcInfo;
import net.sourceforge.czt.typecheck.circus.ErrorAnn;

/**
 * JCircusFrame.java
 *
 * JCircus initial screen.
 *
 * @author Angela
 */
public class JCircusFrame extends JFrame implements ActionListener {
    
    static private final String NEWLINE = "\n";
    
    // This must be an attribute because parser and typechecker may be called
    // more than one time, and the environments must be kept.
    // Think about reinitialization of environments when calling the translation
    // for another input file.
    private JCircus jCircus;

    private JLabel lblInputSpec;
    private JTextField txfInputSpec;
    private JButton btnChooseInput;
    
    private JLabel lblProjName;
    private JTextField txfProjName;
    
    private JLabel lblOutputPath;
    private JTextField txfOutputPath;
    private JButton btnChooseOutput;
    
    private JLabel lblOutputDir;
    private JTextField txfOutputDir;
    
    private JButton btnTranslate;
    
    private JLabel lblLog;
    private JTextArea txaLog;
    
    private boolean canceled = false;
    
    public JCircus getJCircus() {
        return this.jCircus;
    }
    
    public void setCanceled(boolean canceled) {
        this.canceled = canceled;
    }
    
    public JCircusFrame() {
        
        this.jCircus = new JCircus(this);
        
        this.setTitle("JCircus");
        this.setDefaultCloseOperation(JFrame.EXIT_ON_CLOSE);

        //Create and set up the content pane.
        JComponent contentPane = new JPanel();
        contentPane.setOpaque(true); //content panes must be opaque
        contentPane.setBorder(BorderFactory.createEmptyBorder(10, 10, 10, 10));
        this.setContentPane(contentPane);
        
        //Create and set up the layout
        GridBagLayout gridBagLayout = new GridBagLayout();
        contentPane.setLayout(gridBagLayout);
        
        //Create and set up the components
        lblInputSpec = new JLabel("Input specification:");
        txfInputSpec = new JTextField();
        txfInputSpec.setPreferredSize(new Dimension(400, 22));
        btnChooseInput = new JButton("...");
        btnChooseInput.setPreferredSize(new Dimension(30, 22));
        btnChooseInput.addActionListener(this);
        
        lblProjName = new JLabel("Project name:");
        txfProjName = new JTextField();
        txfProjName.setPreferredSize(new Dimension(400, 22));
        txfProjName.getDocument().addDocumentListener(new TextFieldsDocumentListener());
    
        lblOutputPath = new JLabel("Project path:");
        txfOutputPath = new JTextField();
        txfOutputPath.setPreferredSize(new Dimension(400, 22));
        txfOutputPath.getDocument().addDocumentListener(new TextFieldsDocumentListener());
        
        btnChooseOutput = new JButton("...");
        btnChooseOutput.setPreferredSize(new Dimension(30, 22));
        btnChooseOutput.addActionListener(this);
        
        lblOutputDir = new JLabel("Project folders will be created in:");
        txfOutputDir = new JTextField();
        txfOutputDir.setPreferredSize(new Dimension(400, 22));
        txfOutputDir.setEditable(false);

        btnTranslate = new JButton("Translate");
        btnTranslate.setPreferredSize(new Dimension(100, 25));
        btnTranslate.addActionListener(this);
        
        lblLog = new JLabel("Log:");
        txaLog = new JTextArea(5,20);
        txaLog.setMargin(new Insets(5,5,5,5));
        txaLog.setEditable(false);
        JScrollPane logScrollPane = new JScrollPane(txaLog);
        
        //Add components to the content pane
        
        GridBagConstraints c = new GridBagConstraints();
        c.anchor = GridBagConstraints.EAST;

        // First row
        // Input spec - label
        c.gridx = 0;
        c.gridy = 0;
        c.insets = new Insets(1, 1, 1, 1);
        c.fill = GridBagConstraints.NONE;
        c.weightx = 0.0;
        contentPane.add(lblInputSpec, c);

        // Input spec - label
        c.gridx = 1;
        c.gridy = 0;
        c.insets = new Insets(1, 1, 1, 1);
        c.fill = GridBagConstraints.HORIZONTAL;
        c.weightx = 1.0;
        contentPane.add(txfInputSpec, c);
        
        // Input spec - button
        c.gridx = 2;
        c.gridy = 0;
        c.insets = new Insets(1, 1, 1, 1);
        c.fill = GridBagConstraints.HORIZONTAL;
        c.weightx = 0.0;
        contentPane.add(btnChooseInput, c);
        
        // Second row
        // Project name - label
        c.gridx = 0;
        c.gridy = 1;
        c.insets = new Insets(1, 1, 1, 1);
        c.fill = GridBagConstraints.NONE;
        c.weightx = 0.0;
        contentPane.add(lblProjName, c);

        // Input spec - label
        c.gridx = 1;
        c.gridy = 1;
        c.insets = new Insets(1, 1, 1, 1);
        c.fill = GridBagConstraints.HORIZONTAL;
        c.weightx = 1.0;
        contentPane.add(txfProjName, c);
        
        // Third row
        // Output dir - label
        c.gridx = 0;
        c.gridy = 2;
        c.insets = new Insets(1, 1, 1, 1);
        c.fill = GridBagConstraints.NONE;
        c.weightx = 0.0;
        contentPane.add(lblOutputPath, c);

        // Output dir - label
        c.gridx = 1;
        c.gridy = 2;
        c.insets = new Insets(1, 1, 1, 1);
        c.fill = GridBagConstraints.HORIZONTAL;
        c.weightx = 1.0;
        contentPane.add(txfOutputPath, c);
        
        // Output dir - button
        c.gridx = 2;
        c.gridy = 2;
        c.insets = new Insets(1, 1, 1, 1);
        c.fill = GridBagConstraints.HORIZONTAL;
        c.weightx = 0.0;
        contentPane.add(btnChooseOutput, c);
        
        // Rest of components
        c.gridx = 0;
        c.gridy = 3;
        c.fill = GridBagConstraints.HORIZONTAL;
        c.gridwidth = GridBagConstraints.REMAINDER;
        c.insets = new Insets(10, 0, 0, 0);
        contentPane.add(lblOutputDir, c);
        
        c.gridy = GridBagConstraints.RELATIVE;
        c.fill = GridBagConstraints.HORIZONTAL;
        c.gridwidth = GridBagConstraints.REMAINDER;
        c.insets = new Insets(2, 0, 0, 0);
        contentPane.add(txfOutputDir, c);

        c.fill = GridBagConstraints.NONE;
        c.gridwidth = GridBagConstraints.REMAINDER;
        c.anchor = GridBagConstraints.CENTER;
        c.insets = new Insets(10, 0, 0, 0);
        contentPane.add(btnTranslate, c);
        
        c.fill = GridBagConstraints.HORIZONTAL;
        c.gridwidth = GridBagConstraints.REMAINDER;
        c.insets = new Insets(2, 0, 0, 0);
        contentPane.add(lblLog, c);
        
        c.fill = GridBagConstraints.HORIZONTAL;
        c.gridwidth = GridBagConstraints.REMAINDER;
        c.insets = new Insets(2, 0, 0, 0);
        contentPane.add(logScrollPane, c);
        
    }
    
    public void actionPerformed(ActionEvent e) {

        if (e.getSource() == btnChooseInput) {
            btnChooseInputActionPerformed();
        } else if (e.getSource() == btnChooseOutput) {
            btnChooseOutputActionPerformed();
        } else if (e.getSource() == btnTranslate) {
            btnTranslateActionPerformed();
        }

    }

    private void btnChooseInputActionPerformed() {

        //Create a file chooser
        JFileChooser fc = new JFileChooser();
        fc.setFileSelectionMode(JFileChooser.FILES_ONLY);
        fc.setCurrentDirectory(new File(Constants.DIR_EXAMPLES));
        fc.setCurrentDirectory(new File("."));
            
        int returnVal = fc.showOpenDialog(this);

        if (returnVal == JFileChooser.APPROVE_OPTION) {
            // If a file has been chosen
            
            File file = fc.getSelectedFile();
            // Shows the path in the input field
            String path = file.getAbsolutePath();
            txfInputSpec.setText(path);
        }        
    }
    
    private void btnChooseOutputActionPerformed() {

        //Create a file chooser
        JFileChooser fc = new JFileChooser();
        fc.setFileSelectionMode(JFileChooser.DIRECTORIES_ONLY);
        fc.setCurrentDirectory(new File(Constants.DIR_PROJECTS));
        fc.setCurrentDirectory(new File("."));
            
        int returnVal = fc.showOpenDialog(this);

        if (returnVal == JFileChooser.APPROVE_OPTION) {
            // If a file has been chosen
            
            File file = fc.getSelectedFile();
            // Shows the path in the input field
            String path = file.getAbsolutePath();
            txfOutputPath.setText(path);
        }        
    }
    
    private void btnTranslateActionPerformed() {
    
        // clear the log
        txaLog.setDocument(new PlainDocument());
        
        // Reinitializes everything
        this.jCircus = new JCircus(this);
        
        // Checks if the input file has been entered
        String path = txfInputSpec.getText();
        if (path == null || path.equals("")) {
            JOptionPane.showMessageDialog(this, "Enter the input file.");
            return;
        }

        // Checks if the project name has been entered
        String projectName = txfProjName.getText();
        if (projectName == null || projectName.equals("")) {
            JOptionPane.showMessageDialog(this, "Enter the project name.");
            return;
        }

        // Checks if the output path has been entered
        String projectDir = txfOutputPath.getText();
        if (projectDir == null || projectDir.equals("")) {
            JOptionPane.showMessageDialog(this, "Enter the output dir.");
            return;
        }
        
        try {
            
            // Call the translation
            jCircus.jCircusMain(path, projectName, projectDir);
            
        } catch (FailParsingException fpe) {
            
            // Fail parsing
            List errors = fpe.getErrors();
            
            txaLog.append(errors.size() + " parsing error(s)!" + NEWLINE);
            txaLog.setCaretPosition(txaLog.getDocument().getLength());

            for(Object e : errors) {
                txaLog.append(e.toString() + NEWLINE);
                txaLog.setCaretPosition(txaLog.getDocument().getLength());
            }
                
        } catch (FailTypeCheckingException ftce) {
            
            // Type check errors
            List typeErrors = ftce.getErrors();

            txaLog.append(typeErrors.size() + " type error(s)!" + NEWLINE);
            txaLog.setCaretPosition(txaLog.getDocument().getLength());
            
            for(Object e : typeErrors) {
                txaLog.append(e.toString() + NEWLINE);
                txaLog.setCaretPosition(txaLog.getDocument().getLength());
            }                
            
        } catch (FailTranslationException fte) {

            // Type check errors
            List reqErrors = fte.getErrors();

            txaLog.append("Input file does not meet the requirements for translation." + NEWLINE);
            txaLog.append(reqErrors.size() + " error(s)!" + NEWLINE);
            txaLog.setCaretPosition(txaLog.getDocument().getLength());
            
            for(Object e : reqErrors) {
                txaLog.append(e.toString() + NEWLINE);
                txaLog.setCaretPosition(txaLog.getDocument().getLength());
            }                
            
        } catch (JCircusException e) {
            
            JOptionPane.showMessageDialog(this, "Unexpected error. See log for details.", 
                    "Error", JOptionPane.ERROR_MESSAGE);
                    
            try {
                PrintStream printStream = new PrintStream("/log.txt");
                e.printStackTrace(printStream);
                printStream.close();

            } catch (IOException ioe) {

                JOptionPane.showMessageDialog(this, "Can't create log.", 
                        "Error", JOptionPane.ERROR_MESSAGE);
                ioe.printStackTrace();
            } finally {
                e.printStackTrace();
            }

        } catch (Exception e) {
            
            JOptionPane.showMessageDialog(this, "Unexpected error. See log for details.", 
                    "Error", JOptionPane.ERROR_MESSAGE);
                    
            try {
                PrintStream printStream = new PrintStream("/log.txt");
                printStream.print(e.toString());
                printStream.close();

            } catch (IOException ioe) {

                JOptionPane.showMessageDialog(this, "Can't create log.", 
                        "Error", JOptionPane.ERROR_MESSAGE);
                ioe.printStackTrace();
            } finally {
                e.printStackTrace();
            }
        }
        
    }

    /**
     * Reports in the Log area that the parsing has been successfull.
     */
    public void reportParsingOK() {
        txaLog.append("No parsing errors!" + NEWLINE);
    }
    
    /**
     * Reports in the Log area that the type checking has been successfull.
     */
    public void reportTypeCheckingOK() {
        txaLog.append("No type checking errors!" + NEWLINE);
    }

    /**
     * Reports in the Log area that the translation has been successfull.
     */
    public void reportTranslationOK(String projectDir, String projectName) {
        txaLog.append("Translation was successful! Files for project '" + projectName + 
                "' were created at " + getOutputDir(projectDir, projectName) + ".");
    }
    
    /**
     *
     */
    private void changeOutputDir() {
        String outputPath = txfOutputPath.getText();
        String projectName = txfProjName.getText();
        String projectDir = getOutputDir(outputPath, projectName);
        txfOutputDir.setText(projectDir);
    }
    
    /**
     *
     */
    private String getOutputDir(String outputPath, String projectName) {
        return outputPath + "\\" + projectName + "\\" + Constants.DIR_SOURCE;
    }
    
    /**
     * Shows the dialog to select the process for which a class main is
     * to be created.
     *
     */
    public boolean promptForMainProcesses(List<ProcInfo> procInfoList) {
        
        ChooseMainDialog chooseMainDialog = new ChooseMainDialog(this, procInfoList);
        chooseMainDialog.pack();
        chooseMainDialog.setVisible(true);
        
        if (this.canceled == true) {
            JOptionPane.showMessageDialog(this, "Translation canceled by the user.");
            return false;
        } else {
            return true;
        }
    }
    
    /**
     * Shows the dialog to prompt for parameters.
     *
     */
    public boolean promptForParameters(ProcInfo procInfo) {
        
        ParametersDialog ParametersFrame = new ParametersDialog(this, procInfo);
        ParametersFrame.pack();
        ParametersFrame.setVisible(true);
        
        if (this.canceled == true) {
            JOptionPane.showMessageDialog(this, "Translation canceled by the user.");
            return false;
        } else {
            return true;
        }
    }
    
    /**
     * Listener. Listen for changes in the textfields to change the 
     * Project Directory accordingly.
     */
    public class TextFieldsDocumentListener implements DocumentListener {
        
        public void insertUpdate(DocumentEvent e) {
            changeOutputDir();
        }
        
        public void removeUpdate(DocumentEvent e) {
            changeOutputDir();
        }
        
        public void changedUpdate(DocumentEvent e) {
            changeOutputDir();
        }        
    }

    /**
     * Create the GUI and show it.  For thread safety,
     * this method should be invoked from the
     * event-dispatching thread.
     */
    private static void createAndShowGUI() {
        //Make sure we have nice window decorations.
        JFrame.setDefaultLookAndFeelDecorated(true);

        //Create and set up the window.
        JCircusFrame jCircusFrame = new JCircusFrame();
        //jCircusFrame.setUpForTesting();

        //Display the window.
        jCircusFrame.pack();
        jCircusFrame.setVisible(true);
    }

    /**
     * Only for tests.
     */
    public void setUpForTesting() {
        
        this.txfInputSpec.setText("D:\\Angela\\Research\\Java\\JCircus\\src\\examples\\T_MS.tex");
        this.txfProjName.setText("MS_");
        this.txfOutputPath.setText("D:\\Angela\\Research\\Java\\Translator");
        changeOutputDir();
    }
           
    /**
     * Main method.
     */
    public static void main(String[] args) {
        //Schedule a job for the event-dispatching thread:
        //creating and showing this application's GUI.
        javax.swing.SwingUtilities.invokeLater(new Runnable() {
            public void run() {
                createAndShowGUI();
            }
        });
    }    
     
}
